//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Collections;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Properties;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
	/// <summary>
	/// Indicates that the property value of the class is a name reference to a dictionary that can be created through the factory type by name.
	/// </summary>
	[AttributeUsage(AttributeTargets.Class, AllowMultiple = true, Inherited = false)]
    public sealed class ReferenceDictionaryPropertyParameterFactoryAttribute : DictionaryPropertyParameterFactoryAttribute
	{
		/// <summary>
		/// Initialize a new instance of the <see cref="ReferenceDictionaryPropertyParameterAttribute"/> class with the order, the parameter type, the property name, the type of dictionary to create and the property name of the nested element that is the key for the dictionary.
		/// </summary>
		/// <param name="order">The ordinal position of the constructor parameter.</param>
		/// <param name="parameterType">The parameter type.</param>
		/// <param name="propertyName">The name of the property to get the parameter value.</param>
		/// <param name="dictionaryType">The type of dictionary to create.</param>
		/// <param name="keyPropertyName">The property of the nested element that is the key for the dictionary.</param>
		/// <param name="factoryType">The <see cref="INamedObjectFactory"/> used to create the <see cref="IDictionary"/>.</param>
        public ReferenceDictionaryPropertyParameterFactoryAttribute(Type dictionaryType, string keyPropertyName, Type elementFactoryType)
            : base(dictionaryType, keyPropertyName, elementFactoryType)
		{
            if (elementFactoryType == null) throw new ArgumentNullException("factoryType");
            if (!typeof(INamedObjectFactory<object, object>).IsAssignableFrom(elementFactoryType))
			{
				throw new ArgumentException(Resources.ExceptionTypeNotINamedObjectFactory, "factoryType");
			}
		}

        /// <summary>
        /// Gets the <see cref="ReferenceFactory"/> to use to transform the nested element.
		/// </summary>
        /// <returns>The <see cref="ReferenceFactory"/> to use to transform the nested element.</returns>
        protected override IObjectFactory<TObjectToCreate, TObjectConfiguration> GetElementFactory<TObjectToCreate, TObjectConfiguration>()
		{
            return new ReferenceFactory<TObjectToCreate, TObjectConfiguration>(base.GetElementFactory<TObjectToCreate, TObjectConfiguration>());
		}
	}
}
